#ifndef STATISTICAL_EDGES_H
#define STATISTICAL_EDGES_H

#include <opencv2/core/core.hpp>
#include <Eigen/Dense>

namespace statistical_edges {

/**
 * @brief Edge flags activated by the StaticticalDepthEdgeDetector.
 */
enum EdgeFlags {
    WEST = 1,        //!< Indicates wether there is an edge to the left/west of this pixel
    NORTH = 1 << 1,
    EAST = 1 << 2,
    SOUTH = 1 << 3,
    OCCLUDING_WEST = 1 << 4,  //!< Indicates wether the pixel is on a surface border
                              //!< that is occluding the surface to the west/on its left,
                              //!< i.e. the current pixel has a smaller depth than the one
                              //!< to its left.
    OCCLUDING_NORTH = 1 << 5,
    OCCLUDING_EAST = 1 << 6,
    OCCLUDING_SOUTH = 1 << 7,
};

#define DIRECTION_MASK (WEST | NORTH | EAST | SOUTH)
#define OCCLUDING_MASK (OCCLUDING_WEST | OCCLUDING_NORTH | OCCLUDING_EAST | OCCLUDING_SOUTH)

/**
 * @brief Tell if a border pixel is occluding another surface
 */
inline bool is_occluding(uint8_t flags) {
    if (flags & OCCLUDING_MASK)
        return true;
    else return false;
}

/**
 * @brief Tell if a border pixel is occluded by another surface
 */
inline bool is_occluded(uint8_t flags) {
    if (flags && !(flags & OCCLUDING_MASK))
        return true;
    else return false;
}

/**
 * @brief Get a binary mask of a given side of the edge.
 * @param edges Edges detected using the StaticticalDepthEdgeDetector
 * @param occluding_side Extract the occluding side of the edge
 */
cv::Mat1b get_edge_side(cv::Mat1b edges, bool occluding_side = true);

/**
 * @brief Colorize the edge mask return by the StaticticalDepthEdgeDetector.
 *
 * RGB components are defined as:
 * R = 255 if there is an edge on the EAST/WEST axis
 *     0 otherwise
 * B = 255 if there is an edge on the NORTH/SOUTH axis
 *     0 otherwise
 * G = 255 if it is an "occluding edge"
 *     0 otherwise
 *
 * /!\ As OpenCV assumes BGR images, the results can have the red and blue channels swapped.
 */
cv::Mat3b colorize_edges(cv::Mat1b edges);

/**
 * @brief Implementation of the statistical edge detector
 */
class StaticticalDepthEdgeDetector {
public:
    StaticticalDepthEdgeDetector(float prior = 0.1, float threshold = 0.5, int distance = 8);

    /**
     * @brief Constructor that initializes the parameter of the Cauchy distribution.
     * @param image_size The resolution of the depth map that will be given to the detector.
     * @param intrinsic_matrix A 3 by 3 matrix of floating point value encoding the intrinsic parameters of the camera.
     */
    StaticticalDepthEdgeDetector(cv::Size image_size, cv::Mat intrinsic_matrix, float prior = 0.1, float threshold = 0.5, int distance = 8);

    /**
     * @brief Detect the edges in the depth map.
     * @param depth a 2D matrix of float where the depth values are represented in meters.
     * @param noise_std a 2D matrix of float encoding the standard deviation of the noise for each pixel.
     * @param prior The edge prior (number of expected edge pixels in the image divided by the total number of pixels).
     * @param threshold The threshold $\tau$.
     * @param operator_size The spatial support $k$ used to estimate the surface characteriscs.
     * @param features The number of features to include into the model
     */
    cv::Mat1b detect(cv::Mat1f &depth, cv::Mat1f &noise_std, int features = 2);

    cv::Mat1b detect0(cv::Mat1f &depth, cv::Mat1f &noise_std);
    cv::Mat1b detect1(cv::Mat1f &depth, cv::Mat1f &noise_std);
    cv::Mat1b detect2(cv::Mat1f &depth, cv::Mat1f &noise_std);

    /**
     * @brief Set the depth map resolution
     */
    void set_size(cv::Size size);

    /**
     * @brief Set the 3 by 3 matrix of the intrinsic parameter of the camera
     */
    void set_intrinsic_matrix(cv::Mat &mat);

    /**
     * @brief Set a simplified version of the intrinsic parameters of the camera.
     */
    void set_intrinsics(float focal_length_x, float focal_length_y, float principal_point_x, float principal_point_y);

    /**
     * @brief Get/set the edge prior.
     */
    float get_edge_prior();
    void set_edge_prior(float prior);

    /**
     * @brief Get/set the detection threshold.
     */
    float get_detection_threshold();
    void set_detection_threshold(float threshold);

    /**
     * @brief Get/set the distance at which additional points are used
     */
    int get_distance();
    void set_distance(int distance);

    /**
     * @brief Tell if the Cauchy distribution parameters are already precomputed
     */
    bool initialized();

private:
    /**
     * @brief Private method that precompute the parameters of the Cauchy distribution.
     */
    void initialize(cv::Size image_size, cv::Mat intrinsic_matrix);

    /**
     * @brief Private method for computing the various priors from the edge prior.
     */
    void initialize_priors();

    /**
     * @brief Parameters of the Cauchy distributions.
     */
    cv::Mat1f loc_po[2];
    cv::Mat1f loc_pq[2];
    cv::Mat1f loc_qr[2];
    cv::Mat1f loc_qo[2];
    cv::Mat1f loc_or[2];
    cv::Mat1f loc_pr[2];
    cv::Mat1f loc_rr[2];

    cv::Mat1f scale_po[2];
    cv::Mat1f scale_pq[2];
    cv::Mat1f scale_qr[2];
    cv::Mat1f scale_qo[2];
    cv::Mat1f scale_or[2];
    cv::Mat1f scale_pr[2];
    cv::Mat1f scale_rr[2];

    cv::Mat1f cauchy_locations[2];
    cv::Mat1f cauchy_scales[2];

    /**
     * @brief Parameters of the image.
     */
    cv::Size size;
    cv::Mat intrinsic_matrix;

    /**
     * @brief Parameters of the method
     */
    float edge_prior;
    float detection_threshold;
    int distance;

    /**
     * @brief Various priors
     */
    float Spq_prior;
    float Jqr_prior;
    float Sqr_prior;
    float Sqr_Sop_prior;
    float Sqr_Jop_prior;
    float Jop_Jqr_prior;

    Eigen::Matrix<float, 4, 2> A;

    bool _initialized;
};

}

#endif // STATISTICAL_EDGES_H

