
#include <opencv2/imgproc/imgproc.hpp>
#include <statistical_edges.h>

#include "pdfs.h"

using namespace statistical_edges;

cv::Mat1b StaticticalDepthEdgeDetector::detect0(cv::Mat1f &depth, cv::Mat1f &noise_std) {
    if (!initialized())
        initialize(size, intrinsic_matrix);

    double edge_detection_start = (double)cv::getTickCount();

    cv::Mat1b edges = cv::Mat1b::zeros(depth.size());

     // We create a border of the size equal to 1 to handle border cases
    // with values set at the maximum possible float. This will make surface characteristics
    // estimated on the border more unlikely than characteristics estimated on the actual
    // image depth values.
    cv::Mat1f tmp(depth.rows + 2 * 1, depth.cols + 2 * 1);
    cv::copyMakeBorder(depth, tmp,
                       1, 1, 1, 1,
                       cv::BORDER_CONSTANT, cv::Scalar(FLT_MAX));
    cv::Rect roi(1, 1, depth.cols, depth.rows);
    cv::Mat1f depth_padded = tmp(roi);

    // Standard deviation of the noise padded with 0.
    cv::Mat1f tmp2(depth.rows + 2 * 1, depth.cols + 2 * 1);
    cv::copyMakeBorder(noise_std, tmp2,
                       1, 1, 1, 1,
                       cv::BORDER_CONSTANT, cv::Scalar(0));
    cv::Mat1f noise_padded = tmp2(roi);

    // Compute min/max of depth map to determine the distribution of the depth of a single pixel
    float min = FLT_MAX;
    float max = -FLT_MAX;
    float *z_q_ptr = NULL;
    for (int row = depth.rows - 1; row >= 0; row--) {
        z_q_ptr = (float*)depth_padded.ptr(row);
        float *u_q_ptr = (float*)noise_padded.ptr(row);
        for (int col = depth.cols - 1; col >= 0; col--) {
            if (*z_q_ptr != 0.0f) {
                if (*z_q_ptr < min) min = *z_q_ptr;
                if (*z_q_ptr > max) max = *z_q_ptr;

                *u_q_ptr = *u_q_ptr / powf(*z_q_ptr, 2);
                *z_q_ptr = 1.0f / *z_q_ptr;
            }
            z_q_ptr++; u_q_ptr++;
        }
    }


#ifdef __SSE__
    __m128 one_const = _mm_set1_ps(1.0f);
    __m128 zero_const = _mm_set1_ps(0.0f);
    __m128 max_const = _mm_set1_ps(FLT_MAX);
    __m128 p_single_z = _mm_set1_ps(1.0f / (fasterlog(max) - fasterlog(min))); // Reciprocal distribution between min and max
    __m128 threshold = _mm_set1_ps(detection_threshold * edge_prior / ((1.0f - detection_threshold) * (1.0f - edge_prior)));
#else
    // Reciprocal distribution between min and max.
    float p_single_z = 1.0f / (fasterlog(max) - fasterlog(min));
    // Detection threshold.
    float threshold = detection_threshold * edge_prior / ((1 - detection_threshold) * (1 - edge_prior));
#endif

    /**
     * For each pixel, we consider two directions: WEST and NORTH.
     */
    static cv::Point2i dir[2] = {
        cv::Point2i(-1, 0),
        cv::Point2i(0, -1)
    };

    /**
     * Flags
     */
    static uint8_t direction_flags[2][2] = {
        { WEST, EAST },
        { NORTH, SOUTH }
    };

    // Detection loop

    // Pointers to data
    float *locations[2] = {(float *)loc_pq[0].data, (float *)loc_pq[1].data};
    float *scales[2] = {(float *)scale_pq[0].data, (float *)scale_pq[1].data};
    uint8_t *edge_ptr = edges.data;
    uint8_t *edge2_ptr[2] = {edge_ptr - 1, edge_ptr - edges.step1()};

    for (int row = 0; row < depth.rows; row++) {

        // Pointers to row data
        float *depth_ptr = (float *)depth_padded.ptr(row);
        float *depth2_ptr[2] = {depth_ptr - 1, depth_ptr - depth_padded.step1()}; // WEST and NORTH depth values
        float *noise_ptr = (float *)noise_padded.ptr(row);
        float *noise2_ptr[2] = {noise_ptr - 1, noise_ptr - noise_padded.step1()};

#ifndef __SSE__
        // Unoptimized version
        for (int col = 0; col< depth.cols; col++) {
            float d1 = *(depth_ptr++);

            // For each directions, we compute the surface and edge probability and apply the decision rule
            for (int i = 0; i < 2; i++) {
                float d2 = *(depth2_ptr[i]++);

                float p_surface = 0.0f;

                if (d1 == 0.0f) {
                    // Special case if we do not have any depth information! There is an edge only if
                    // d2 has a depth!
                    if (d2 == 0.0f) { p_surface = FLT_MAX; }
                    locations[i]++;
                    scales[i]++;
                    noise2_ptr[i]++;
                }
                else {
                    // Retrieve the parameter of the Cauchy
                    float loc = d1 * *(locations[i]++);
                    float scale = d1 * *(scales[i]++);

                    float u = sqrt(powf(*noise_ptr, 2) + powf(*(noise2_ptr[i]++), 2));

                    // Surface probability without surface characteristics
                    float p_depth = pseudo_voigt(d2, loc, u, scale);

                    // We aggregate the probabilities
                    p_surface = p_depth;
                }

                // Bayes decision rule
                if (p_surface / p_single_z - threshold < 0.0f) {
                    // Edge detected!
                    *edge_ptr |= direction_flags[i][0];

                    // If the other pixel is within image boundaries, we set its corresponding
                    // flags into the edge map.
                    if (!((i == 0 && col == 0) || (i == 1 && row == 0))) {
                        *edge2_ptr[i] |= direction_flags[i][1];

                        if (d1 > d2) {
                            // d2 is closer than d1 and thus, d2 occludes d1.
                            *edge2_ptr[i] |= direction_flags[i][1] << 4;
                        }
                        else {
                            // d1 is closer than d2 and thus, d1 occludes d2.
                            *edge_ptr |= direction_flags[i][0] << 4;
                        }
                    }
                }
                edge2_ptr[i]++;
            }

            noise_ptr++;
            edge_ptr++;
        }
#else
        // Optimized SSE version
        for (int col = 0; col< depth.cols; col += 4) {
            __m128 d1 = _mm_loadu_ps(depth_ptr);
            __m128 d1_valid = _mm_cmpneq_ps(d1, zero_const);
            depth_ptr += 4;

            // For each direction
            for (int i = 0; i < 2; i++) {
                __m128 d2 = _mm_loadu_ps(depth2_ptr[i]);
                __m128 d2_valid = _mm_cmpneq_ps(d2, zero_const);
                __m128 d1d2_valid = _mm_and_ps(d1_valid, d2_valid);
                __m128 d1d2_nonvalid = _mm_cmpeq_ps(_mm_or_ps(d1_valid, d2_valid), zero_const);

                // Retrieve the parameter of the Cauchy
                __m128 loc = _mm_load_ps(locations[i]);
                loc = _mm_mul_ps(loc, d1);
                __m128 scale = _mm_load_ps(scales[i]);
                scale = _mm_mul_ps(scale, d1);

                __m128 u1 = _mm_loadu_ps(noise_ptr);
                __m128 u2 = _mm_loadu_ps(noise2_ptr[i]); noise2_ptr[i] += 4;
                __m128 u = _mm_sqrt_ps(_mm_add_ps(_mm_mul_ps(u1, u1), _mm_mul_ps(u2, u2)));

                __m128 p_depth = pseudo_voigt_sse(d2, loc, u, scale);

                __m128 p_surface = p_depth;
                p_surface = _mm_or_ps(p_surface, _mm_and_ps(d1d2_nonvalid, max_const));

                // Bayes decision rule
                __m128 prob_div = _mm_div_ps(p_surface, p_single_z);
                __m128 comp = _mm_cmplt_ps(_mm_sub_ps(prob_div, threshold), zero_const);
                __m128 depth_comp = _mm_cmpgt_ps(d1, d2);

                for (int j = 0; j < 4; j++) {
                    if (comp[j] != 0xFFFFFFFF)
                        continue;

                    // Edge detected!
                    edge_ptr[j] |= direction_flags[i][0];

                    if (!((i == 0 && col == 0) || (i == 1 && row == 0))) {
                        edge2_ptr[i][j] |= direction_flags[i][1];
                        if (depth_comp[j] == 0xFFFFFFFF) {
                            edge2_ptr[i][j] |= direction_flags[i][1] << 4;
                        }
                        else {
                            edge_ptr[j] |= direction_flags[i][0] << 4;
                        }
                    }
                }
                depth2_ptr[i] += 4;
                locations[i] += 4;
                scales[i] += 4;
                edge2_ptr[i] += 4;
            }

            noise_ptr += 4;
            edge_ptr += 4;
        }
#endif
    }

    double edge_detection_time = ((double)cv::getTickCount() - edge_detection_start ) / cv::getTickFrequency();
    printf ("Edge detection took %fms\n", edge_detection_time * 1000);

    return edges;
}