
#include <opencv2/imgproc/imgproc.hpp>

#include <statistical_edges.h>

using namespace statistical_edges;

#define absmin(a, b) (fabs(a) < fabs(b) ? (a) : (b))

cv::Mat1b StaticticalDepthEdgeDetector::detect(cv::Mat1f &depth, cv::Mat1f &noise_std, int features) {
    if (features == 0)
        return detect0 (depth, noise_std);
    else if (features == 1)
        return detect1 (depth, noise_std);
    else if (features == 2)
        return detect2 (depth, noise_std);
}

void StaticticalDepthEdgeDetector::initialize(cv::Size s, cv::Mat intrinsic_matrix) {
    double init_start_time = (double)cv::getTickCount();

    for (int i = 0; i < 2; i++) {
        loc_po[i] = cv::Mat1f(s);
        loc_pq[i] = cv::Mat1f(s);
        loc_qr[i] = cv::Mat1f(s);
        loc_qo[i] = cv::Mat1f(s);
        loc_or[i] = cv::Mat1f(s);
        loc_pr[i] = cv::Mat1f(s);
        loc_rr[i] = cv::Mat1f(s);

        scale_po[i] = cv::Mat1f(s);
        scale_pq[i] = cv::Mat1f(s);
        scale_qr[i] = cv::Mat1f(s);
        scale_qo[i] = cv::Mat1f(s);
        scale_or[i] = cv::Mat1f(s);
        scale_pr[i] = cv::Mat1f(s);
        scale_rr[i] = cv::Mat1f(s);
    }

    cv::Mat1f locs[7][2] = {loc_po, loc_pq, loc_qr, loc_qo, loc_or, loc_pr, loc_rr};
    cv::Mat1f scales[7][2] = {scale_po, scale_pq, scale_qr, scale_qo, scale_or, scale_pr, scale_rr};
    int idx_diff[7] = {-1, -1, 0, 0, -1 - distance, -1, distance - 1};
    int k[7] = {-distance, 1, distance, - distance - 1, 2 * distance + 1, distance + 1, 1};

    cv::Mat1f inv_intrinsics = intrinsic_matrix.inv();

    cv::Mat1f p[2];
    p[0] = cv::Mat1f(3,1); p[0](2) = 1;
    p[1] = cv::Mat1f(3,1); p[1](2) = 1;
    cv::Mat1f q[2];
    q[0] = cv::Mat1f(3,1); q[0](2) = 1;
    q[1] = cv::Mat1f(3,1); q[1](2) = 1;

    for (int row = s.height - 1; row >= 0; row--) {
        for (int col = s.width - 1; col >= 0; col--) {
            for (int i = 0; i < 7; i++) {
                p[0](0) = col + idx_diff[i]; p[0](1) = row;
                p[1](0) = col; p[1](1) = row + idx_diff[i];
                q[0](0) = p[0](0) + k[i]; q[0](1) = row;
                q[1](0) = col; q[1](1) = p[1](1) + k[i];


                for (int j = 0; j < 2; j++) {
                    cv::Mat1f p_plus_q = 0.5 * inv_intrinsics * (p[j] + q[j]);
                    cv::Mat1f p_minus_q = 0.5 * inv_intrinsics * (p[j] - q[j]);

                    float norm_p_plus_q = p_plus_q.dot(p_plus_q);
                    float norm_p_minus_q = p_minus_q.dot(p_minus_q);

                    float a = - (p_plus_q.dot(p_minus_q)) / norm_p_plus_q;
                    float b = sqrt(norm_p_minus_q / norm_p_plus_q - a * a);

                    float loc = (1 - powf(a, 2) - powf(b, 2)) / (powf(1 + a, 2) + powf(b, 2));
                    float scale = 2 * b / (powf(1 + a, 2) + powf(b, 2));

                    locs[i][j](row, col) = loc;
                    scales[i][j](row, col) = std::abs(scale);
                }
            }

        }
    }

    A << 1, 0,
         1.0f - distance / (2*distance+1.0f), distance / (2*distance + 1.0f),
         1.0f - (distance + 1.0f) / (2 * distance + 1.0f), (distance + 1.0f) / (2 * distance + 1.0f),
         0, 1;

    double init_time = ((double)cv::getTickCount() - init_start_time ) / cv::getTickFrequency();
    printf ("Initialization took %fms\n", init_time * 1000);

    _initialized = true;
}

cv::Mat1b statistical_edges::get_edge_side(cv::Mat1b edges, bool occluding_side) {
    cv::Mat1b ans = cv::Mat1b::zeros(edges.size());

    for (int row = edges.rows - 1; row >= 0; row--) {
        for (int col = edges.cols - 1; col >= 0; col--) {
            if (occluding_side && is_occluding(edges(row, col)))
                ans(row, col) = 255;
            else if (!occluding_side && is_occluded(edges(row, col)))
                ans(row, col) = 255;
        }
    }
    return ans;
}

cv::Mat3b statistical_edges::colorize_edges(cv::Mat1b edges) {
    cv::Mat3b ans = cv::Mat1b::zeros(edges.size());

    for (int row = edges.rows - 1; row >= 0; row--) {
        for (int col = edges.cols - 1; col >= 0; col--) {
            cv::Vec3b color;
            if (edges(row, col) & (WEST | EAST))
                color[0] = 255;
            if (edges(row, col) & (NORTH | SOUTH))
                color[2] = 255;
            if (is_occluding(edges(row, col)))
                color[1] = 255;
            ans(row, col) = color;
        }
    }
    return ans;
}

StaticticalDepthEdgeDetector::StaticticalDepthEdgeDetector(float _prior, float _threshold, int _distance)
: edge_prior(_prior), detection_threshold(_threshold), distance(_distance),
  size(cv::Size(-1, -1)), _initialized(false) {
    initialize_priors();
}

StaticticalDepthEdgeDetector::StaticticalDepthEdgeDetector(cv::Size size, cv::Mat intrinsic_matrix, float _prior, float _threshold, int _distance)
: edge_prior(_prior), detection_threshold(_threshold), distance(_distance)
{
    initialize(size, intrinsic_matrix);
    initialize_priors();
}

void StaticticalDepthEdgeDetector::set_size(cv::Size size) {
    if (this->size != size) {
        this->size = size;
        _initialized = false;
    }
}

void StaticticalDepthEdgeDetector::set_intrinsic_matrix(cv::Mat &mat) {
    this->intrinsic_matrix = mat;
    _initialized = false;
}

void StaticticalDepthEdgeDetector::set_intrinsics(float focal_length_x, float focal_length_y, float principal_point_x, float principal_point_y) {
    cv::Mat1f mat = cv::Mat1f::eye(3, 3);
    mat(0, 0) = focal_length_x;
    mat(1, 1) = focal_length_x;
    mat(0, 2) = principal_point_x;
    mat(1, 2) = principal_point_y;

    set_intrinsic_matrix(mat);
}

float StaticticalDepthEdgeDetector::get_edge_prior() {
    return edge_prior;
}

void StaticticalDepthEdgeDetector::set_edge_prior(float prior) {
    if (edge_prior != prior) {
        edge_prior = prior;
        initialize_priors();
    }
}

float StaticticalDepthEdgeDetector::get_detection_threshold() {
    return detection_threshold;
}

void StaticticalDepthEdgeDetector::set_detection_threshold(float threshold) {
    detection_threshold = threshold;
}

int StaticticalDepthEdgeDetector::get_distance() {
    return distance;
}

void StaticticalDepthEdgeDetector::set_distance(int _distance) {
    if (distance != _distance) {
        _initialized = false;
        distance = _distance;
        initialize_priors();
    }
}

bool StaticticalDepthEdgeDetector::initialized() {
    return _initialized;
}

void StaticticalDepthEdgeDetector::initialize_priors() {
    Spq_prior = 1 - edge_prior;
    Sqr_prior = powf(Spq_prior, distance);
    Jqr_prior = 1.0f - Sqr_prior;
    Sqr_Sop_prior = Sqr_prior * Sqr_prior;
    Sqr_Jop_prior = Sqr_prior * Jqr_prior;
    Jop_Jqr_prior = Jqr_prior * Jqr_prior;
}
