
#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>

#include <iostream>
#include <string>

#include <statistical_edges.h>

#define NOISE_SCALE 0.002 // For the Microsoft Kinect 1

void print_usage(const char *program_name) {
    std::cout << "Usage : 1) " << program_name << " YML_DEPTH_MAP [THRESHOLD = 0.5] [PRIOR = 0.1] [SPATIAL SUPPORT = 8] [PED version = 2]" << std::endl;
    std::cout << "        2) " << program_name << " PNG_DEPTH_MAP FX FY PX PY [THRESHOLD = 0.5] [PRIOR = 0.1] [SPATIAL SUPPORT = 8] [PED version = 2]" << std::endl << std::endl;

    std::cout << "The program supports two input formats:" << std::endl;
    std::cout << " 1) YML file saved by OpenCV where the depth has the 'depth' key,\n"
                 "    the noise standard deviation has the 'uncertainty' key and\n"
                 "    the intrinsics parameters of the camera are given by the keys\n"
                 "    'fx', 'fy', 'px' and 'py' for the focal lengths and principal points." << std::endl;
    std::cout << " 2) 16 bits PNG file where the depth is encoded in millimeters.\n"
                 "    In this case, the noise standard deviation is computed as:\n"
                 "     noise_std = [NOISE_SCALE=" << NOISE_SCALE << "] * depth * depth,\n"
                 "    where NOISE_SCALE can be changed in the detect_edges.cpp file.\n"
                 "    The intrinsics parameters must be given after the path to the image." << std::endl << std::endl;

    std::cout << "The edge detector parameters comes after the input specification and are optional."
        << std::endl << std::endl;

    std::cout << "The program outputs 4 files:" << std::endl;
    std::cout << " [edges.png] The detected edge map as given by our method" << std::endl;
    std::cout << " [binary_edges.png] Binary edge map." << std::endl;
    std::cout << " [depth_gray.png] normalized gray-scale version of the depth map;" << std::endl;
    std::cout << " [depth_rgb.png] normalized RGB version of the depth map." << std::endl << std::endl;

}

cv::Mat1b get_gray_depth(cv::Mat1f &depth);
cv::Mat3b get_rgb_depth(cv::Mat1f &depth);
void open_yaml(std::string file, cv::Mat1f &depth, cv::Mat1f &noise_std, float &fx, float &fy, float &px, float &py);

int main(int argc, char **argv) {
    if (argc < 2) {
        print_usage(argv[0]);
        exit(EXIT_FAILURE);
    }

    std::string input_depth_map = argv[1];

    float threshold = 0.5f;
    float prior = 0.1f;
    int spatial_support = 8;
    int ped = 2;

    float fx, fy, px, py;

    cv::Mat1f depth;
    cv::Mat1f noise_std;

    if (strcasecmp(input_depth_map.substr(input_depth_map.length() - 3, std::string::npos).c_str(), "png") == 0) {
        if (argc < 6) {
            std::cerr << "Error: intrinsic parameters not given" << std::endl;
        }
        fx = atof(argv[2]);
        fy = atof(argv[3]);
        px = atof(argv[4]);
        py = atof(argv[5]);

        // Fetch method parameters
        if (argc >= 7) threshold = atof(argv[6]);
        if (argc >= 8) prior = atof(argv[7]);
        if (argc >= 9) spatial_support = atoi(argv[8]);
        if (argc >= 10) ped = atoi(argv[9]);

        cv::Mat_<uint16_t> depth_mm = cv::imread(input_depth_map, CV_LOAD_IMAGE_ANYDEPTH);
        if (depth_mm.empty()) {
            std::cerr << "Error: could not open PNG file '" << input_depth_map << "'!" << std::endl;
            exit(EXIT_FAILURE);
        }
        depth_mm.convertTo(depth, depth.type(), 0.001f);
        noise_std = NOISE_SCALE * depth.mul(depth);
    }
    else {
        if (argc >= 3) threshold = atof(argv[2]);
        if (argc >= 4) prior = atof(argv[3]);
        if (argc >= 5) spatial_support = atoi(argv[4]);
        if (argc >= 6) ped = atoi(argv[5]);

        open_yaml(input_depth_map, depth, noise_std, fx, fy, px, py);
        noise_std = NOISE_SCALE * noise_std;
    }

    printf ("Parameters :\n");
    printf (" - threshold = %f\n", threshold);
    printf (" - prior = %f\n", prior);
    printf (" - spatial support = %d\n", spatial_support);
    printf (" - PED version = %d\n", ped);


    statistical_edges::StaticticalDepthEdgeDetector detector;
    detector.set_size(depth.size());
    detector.set_intrinsics(fx, fy, px, py);
    detector.set_detection_threshold(threshold);
    detector.set_distance(spatial_support);
    detector.set_edge_prior(prior);

    cv::Mat1b edges = detector.detect(depth, noise_std, ped);

    cv::imwrite("edges.png", edges);
    cv::imwrite("binary_edges.png", edges > 0);
    cv::Mat1b gray_depth = get_gray_depth(depth);
    cv::imwrite("depth_gray.png", gray_depth);
    cv::Mat3b rgb_depth = get_rgb_depth(depth);
    cv::imwrite("depth_rgb.png", rgb_depth);

    cv::imshow("RGB depth", rgb_depth);
    cv::imshow("Gray-level depth", gray_depth);
    cv::imshow("Edges", statistical_edges::colorize_edges(edges));

    cv::waitKey(0);

    return 0;
}

cv::Mat1b get_gray_depth(cv::Mat1f &depth) {
    cv::Mat1b gray_depth = cv::Mat1b::zeros(depth.size());

    float min = FLT_MAX;
    float max = -FLT_MAX;

    for (int row = depth.rows - 1; row >= 0; row--) {
        for (int col = depth.cols - 1; col >= 0; col--) {
            if (depth(row, col) < min) min = depth(row, col);
            if (depth(row, col) > max) max = depth(row, col);
        }
    }

    for (int row = depth.rows - 1; row >= 0; row--) {
        for (int col = depth.cols - 1; col >= 0; col--) {
            gray_depth(row, col) = 255 * (depth(row, col) - min) / (max - min);
        }
    }

    return gray_depth;
}

cv::Mat3b get_rgb_depth(cv::Mat1f &depth) {
    cv::Mat3b rgb_depth = cv::Mat3b::zeros(depth.size());

    float min = FLT_MAX;
    float max = -FLT_MAX;

    for (int row = depth.rows - 1; row >= 0; row--) {
        for (int col = depth.cols - 1; col >= 0; col--) {
            if (depth(row, col) < min) min = depth(row, col);
            if (depth(row, col) > max) max = depth(row, col);
        }
    }

    for (int row = depth.rows - 1; row >= 0; row--)
    {
        unsigned char *rgb = rgb_depth.row(row).ptr(0);
        for (int col = 0; col < depth.cols; col++)
        {
            int val = 255 * 6 * (depth.at<float>(row,col) - min) / (max - min);
            if (val < 0)
                val = 0;

            int lb = val & 0xff;
            switch (val >> 8)
            {
            case 0:
                *rgb++ = 255;
                *rgb++ = 255 - lb;
                *rgb++ = 255 - lb;
                break;
            case 1:
                *rgb++ = 255;
                *rgb++ = lb;
                *rgb++ = 0;
                break;
            case 2:
                *rgb++ = 255 - lb;
                *rgb++ = 255;
                *rgb++ = 0;
                break;
            case 3:
                *rgb++ = 0;
                *rgb++ = 255;
                *rgb++ = lb;
                break;
            case 4:
                *rgb++ = 0;
                *rgb++ = 255 - lb;
                *rgb++ = 255;
                break;
            case 5:
                *rgb++ = 0;
                *rgb++ = 0;
                *rgb++ = 255 - lb;
                break;
            default:
                *rgb++ = 0;
                *rgb++ = 0;
                *rgb++ = 0;
            }
        }
    }

    return rgb_depth;
}

void open_yaml(std::string file, cv::Mat1f &depth, cv::Mat1f &noise_std, float &fx, float &fy, float &px, float &py) {
    cv::FileStorage fs;
    fs.open(file, cv::FileStorage::READ);

    if (!fs.isOpened()) {
        std::cerr << "Error: could not open YAML file '" << file << "'!" << std::endl;
        exit(EXIT_FAILURE);
    }

    bool error = false;
    cv::FileNode fn = fs["depth"];
    if (!fn.empty()) {
        fn >> depth;
    }
    else {
        std::cerr << "Error: no depth key in YAML file '" << file << "'!" << std::endl;
        error = true;
    }

    fn = fs["uncertainty"];
    if (!fn.empty())
        fn >> noise_std;
    else {
        std::cerr << "Error: no uncertainty key in YAML file '" << file << "'!" << std::endl;
        error = true;
    }


    fn = fs["fx"];
    if (!fn.empty())
        fn >> fx;
    else {
        std::cerr << "Error: no fx key in YAML file '" << file << "'!" << std::endl;
        error = true;
    }
    fn = fs["fy"];
    if (!fn.empty())
        fn >> fy;
    else {
        std::cerr << "Error: no fy key in YAML file '" << file << "'!" << std::endl;
        error = true;
    }
    fn = fs["px"];
    if (!fn.empty())
        fn >> px;
    else {
        std::cerr << "Error: no px key in YAML file '" << file << "'!" << std::endl;
        error = true;
    }
    fn = fs["py"];
    if (!fn.empty())
        fn >> py;
    else {
        std::cerr << "Error: no py key in YAML file '" << file << "'!" << std::endl;
        error = true;
    }

    if (error)
        exit (EXIT_FAILURE);
}
