
#include <opencv2/highgui/highgui.hpp>
#include <statistical_edges.h>

using namespace statistical_edges;

int main(int argc, char **argv) {
    printf ("Usage: %s [threshold (default 0.5)] [prior (default 0.1)] [operator size (default 8)]\n", argv[0]);

    float threshold = argc >= 2 ? atof(argv[1]) : 0.5;
    float prior = argc >= 3 ? atof(argv[2]) : 0.1;
    float distance = argc >= 4 ? atof(argv[3]) : 8;

    printf("Parameters :\n");
    printf(" - threshold = %f\n", threshold);
    printf(" - prior = %f\n", prior);
    printf(" - distance = %f\n", distance);

    cv::VideoCapture cap(CV_CAP_OPENNI);
    if (!cap.isOpened()) {
        printf ("No camera detected!");
    }

    int width = cap.get(CV_CAP_PROP_FRAME_WIDTH);
    int height = cap.get(CV_CAP_PROP_FRAME_HEIGHT);
    cv::Size size(width, height);

    // Setup intrinsic matrix
    float focal_length = cap.get(CV_CAP_PROP_OPENNI_FOCAL_LENGTH);
    cv::Mat1f K = cv::Mat1f::eye(3, 3);
    K(0, 0) = K(1, 1) = focal_length;
    K(0, 2) = width / 2.0f;
    K(1, 2) = width / 2.0f;

    StaticticalDepthEdgeDetector detector(size, K, prior, threshold, distance);

    for(;;) {
        cv::Mat depth_16uc1;
        cap >> depth_16uc1;

        // Obtain depth in meters
        cv::Mat1f depth;
        depth_16uc1.convertTo(depth, depth.type(), 1.0f / 1000.0f);

        cv::Mat1b depth_normalized(depth.size());
        cv::normalize(depth, depth_normalized, 0, 255, cv::NORM_MINMAX, depth_normalized.type());

        // noise standard deviation is 0.0019 times the depth squared
        cv::Mat1f noise = 0.0019 * depth.mul(depth);

        cv::Mat1b edges = detector.detect(depth, noise);

        cv::imshow("depth", depth_normalized);
        cv::imshow("edges", edges > 0);
        cv::imshow("Closest border", get_edge_side(edges));

        if (cv::waitKey(10) >= 0)
            break;
    }
}
