/* 
 * Based on openni_narf_keypoint_extraction tutorials from Point Cloud Library
 * Original author: Bastian Steder
 */

/* ---[ */
#include <iostream>
using namespace std;
#include <pcl/io/openni_grabber.h>
#include <pcl/range_image/range_image_planar.h>
#include <pcl/common/common_headers.h>
#include <pcl/visualization/range_image_visualizer.h>
#include <pcl/visualization/pcl_visualizer.h>

#include <statistical_edges.h>

std::string device_id = "#1";

float angular_resolution = 0.5;
float support_size = 0.2f;
bool set_unseen_to_max_range = true;
int max_no_of_threads = 1;
float min_interest_value = 0.5;

float threshold = 0.5;
float prior = 0.1;
int feature_distance = 8;
int features = 2; // 0 1 or 2

boost::mutex depth_image_mutex,
             ir_image_mutex,
             image_mutex;
pcl::PointCloud<pcl::PointXYZ>::ConstPtr point_cloud_ptr;
boost::shared_ptr<openni_wrapper::DepthImage> depth_image_ptr;
boost::shared_ptr<openni_wrapper::IRImage> ir_image_ptr;
boost::shared_ptr<openni_wrapper::Image> image_ptr;

bool received_new_depth_data = false,
     received_new_ir_image   = false,
     received_new_image   = false;
struct EventHelper
{
  void
  depth_image_cb (const boost::shared_ptr<openni_wrapper::DepthImage>& depth_image)
  {
    if (depth_image_mutex.try_lock ())
    {
      depth_image_ptr = depth_image;
      depth_image_mutex.unlock ();
      received_new_depth_data = true;
    }
  }
};


int main (int argc, char** argv)
{  
  pcl::visualization::RangeImageVisualizer range_image_widget ("Range Image");
  
  pcl::visualization::PCLVisualizer viewer ("3D Viewer");
  viewer.addCoordinateSystem (1.0f, "global");
  viewer.setBackgroundColor (1, 1, 1);
  
  viewer.initCameraParameters ();
  viewer.setCameraPosition (0.0, -0.3, -2.0,
                            0.0, -0.3, 1.0,
                            0.0, -1.0, 0.0);
  
  openni_wrapper::OpenNIDriver& driver = openni_wrapper::OpenNIDriver::getInstance ();
  if (driver.getNumberDevices () > 0)
  {
    for (unsigned deviceIdx = 0; deviceIdx < driver.getNumberDevices (); ++deviceIdx)
    {
      cout << "Device: " << deviceIdx + 1 << ", vendor: " << driver.getVendorName (deviceIdx)
           << ", product: " << driver.getProductName (deviceIdx) << ", connected: "
           << (int) driver.getBus (deviceIdx) << " @ " << (int) driver.getAddress (deviceIdx)
           << ", serial number: \'" << driver.getSerialNumber (deviceIdx) << "\'\n";
    }
  }
  else
  {
    cout << "\nNo devices connected.\n\n";
    return 1;
  }
  
  pcl::Grabber* interface = new pcl::OpenNIGrabber (device_id);
  EventHelper event_helper;
  
  boost::function<void (const boost::shared_ptr<openni_wrapper::DepthImage>&) > f_depth_image =
    boost::bind (&EventHelper::depth_image_cb, &event_helper, _1);
  boost::signals2::connection c_depth_image = interface->registerCallback (f_depth_image);
  
  cout << "Starting grabber\n";
  interface->start ();
  cout << "Done\n";
  
  boost::shared_ptr<pcl::RangeImagePlanar> range_image_planar_ptr (new pcl::RangeImagePlanar);
  pcl::RangeImagePlanar& range_image_planar = *range_image_planar_ptr;
  
  pcl::PointCloud<pcl::PointXYZ>::Ptr keypoints_cloud_ptr (new pcl::PointCloud<pcl::PointXYZ>);
  pcl::PointCloud<pcl::PointXYZ>& keypoints_cloud = *keypoints_cloud_ptr;
  
  cv::Mat1f range_image_cv;
  cv::Mat1f noise;
  statistical_edges::StaticticalDepthEdgeDetector edge_detector;
  
  double total_edge_detection_time = 0.0;
  int edge_detection_performed = 0;
  
  while (!viewer.wasStopped ())
  {
    viewer.spinOnce ();
    range_image_widget.spinOnce ();  // process GUI events
    pcl_sleep (0.01);
    
    bool got_new_range_image = false;
    if (received_new_depth_data && depth_image_mutex.try_lock ())
    {
      received_new_depth_data = false;
      
      //unsigned long time_stamp = depth_image_ptr->getTimeStamp ();
      //int frame_id = depth_image_ptr->getFrameID ();
      const unsigned short* depth_map = depth_image_ptr->getDepthMetaData ().Data ();
      int width = depth_image_ptr->getWidth (), height = depth_image_ptr->getHeight ();
      float center_x = width/2, center_y = height/2;
      float focal_length_x = depth_image_ptr->getFocalLength (), focal_length_y = focal_length_x;
      float original_angular_resolution = asinf (0.5f*float (width)/float (focal_length_x)) / (0.5f*float (width));
      float desired_angular_resolution = angular_resolution;
      range_image_planar.setDepthImage (depth_map, width, height, center_x, center_y,
                                        focal_length_x, focal_length_y, desired_angular_resolution);
      
      const cv::Mat range_image_16u_cv(height, width, CV_16UC1, (void *)depth_map);
      range_image_16u_cv.convertTo(range_image_cv, range_image_cv.type(), 0.001);
      noise = 0.002 * range_image_cv.mul(range_image_cv);
      
      depth_image_mutex.unlock ();
      got_new_range_image = !range_image_planar.points.empty ();
      
      if (!edge_detector.initialized()) {
        cout << "Initializing edge detector..." << std::endl;
        edge_detector.set_intrinsics(focal_length_x, focal_length_y, center_x, center_y);
        edge_detector.set_size(cv::Size(width, height));
        edge_detector.set_detection_threshold(threshold);
        edge_detector.set_distance(feature_distance);
        edge_detector.set_edge_prior(prior);
        cout << "done" << std::endl;
      }
    }
    
    if (!got_new_range_image)
      continue;
   
    // Detect edges
    double edge_detection_start_time = pcl::getTime();
    cv::Mat1b edges = edge_detector.detect(range_image_cv, noise, features);
    double edge_detection_time = pcl::getTime() - edge_detection_start_time;
    total_edge_detection_time += edge_detection_time;
    edge_detection_performed++;
    std::cout << "Edge detection took " << 1000.0 * edge_detection_time << "ms. Avg = " << 1000.0 * total_edge_detection_time / edge_detection_performed << "ms.\n";
    
    std::vector<int> occluding_indices;
    std::vector<int> occluded_indices;
    for (int row = edges.rows - 1; row >= 0; row--) {
        for (int col = edges.cols - 1; col >= 0; col--) {
            //printf ("Depth (%d, %d) = %f\n", row, col, range_image_cv(row, col));
            if (statistical_edges::is_occluding(edges(row, col)))
                occluding_indices.push_back(row * edges.cols + col);
            if (statistical_edges::is_occluded(edges(row, col)))
                occluded_indices.push_back(row * edges.cols + col);
        }
    }
    
    printf ("Occluding edges: %d\n", occluding_indices.size());
    printf ("Occluded edges: %d\n", occluded_indices.size());

   
    // ----------------------------------------------
    // -----Show keypoints in range image widget-----
    // ----------------------------------------------
    range_image_widget.showRangeImage (range_image_planar, 0.5f, 10.0f);
    //for (size_t i=0; i<keypoint_indices.points.size (); ++i)
      //range_image_widget.markPoint (keypoint_indices.points[i]%range_image_planar.width,
                                    //keypoint_indices.points[i]/range_image_planar.width,
                                    //pcl::visualization::Vector3ub (0,255,0));

    pcl::PointCloud<pcl::PointXYZ>::Ptr occluding_point_cloud(new pcl::PointCloud<pcl::PointXYZ>);
    pcl::PointCloud<pcl::PointXYZ>::Ptr occluded_point_cloud(new pcl::PointCloud<pcl::PointXYZ>);
    
    pcl::copyPointCloud(range_image_planar, occluding_indices, *occluding_point_cloud);
    pcl::copyPointCloud(range_image_planar, occluded_indices, *occluded_point_cloud);
    
    // -------------------------------------
    // -----Show keypoints in 3D viewer-----
    // -------------------------------------
    pcl::visualization::PointCloudColorHandlerCustom<pcl::PointWithRange> color_handler_cloud
      (range_image_planar_ptr, 0, 0, 0);
    if (!viewer.updatePointCloud<pcl::PointWithRange> (range_image_planar_ptr, color_handler_cloud, "range image"))
      viewer.addPointCloud<pcl::PointWithRange> (range_image_planar_ptr, color_handler_cloud, "range image");
    viewer.setPointCloudRenderingProperties (pcl::visualization::PCL_VISUALIZER_POINT_SIZE, 1, "range image");

    /*if (!viewer.updatePointCloud (occluding_point_cloud, "occluding edges"))
      viewer.addPointCloud (occluding_point_cloud, "occluding edges");
    viewer.setPointCloudRenderingProperties (pcl::visualization::PCL_VISUALIZER_POINT_SIZE, 2, "occluding edges");
    viewer.setPointCloudRenderingProperties (pcl::visualization::PCL_VISUALIZER_COLOR, 0.0f, 1.0f, 0.0f, "occluding edges");
    
    if (!viewer.updatePointCloud (occluded_point_cloud, "occluded edges"))
      viewer.addPointCloud (occluded_point_cloud, "occluded edges");
    viewer.setPointCloudRenderingProperties (pcl::visualization::PCL_VISUALIZER_POINT_SIZE, 2, "occluded edges");
    viewer.setPointCloudRenderingProperties (pcl::visualization::PCL_VISUALIZER_COLOR, 0.0f, 1.0f, 0.0f, "occluded edges");*/
  }

  interface->stop ();
}
